import numpy

def rgb2gray(image, weights=(0.2125, 0.7154, 0.0721)):
    """Converts a color image with three channels (RGB) to grayscale.

    The given RGB color image is converted to grayscale by summing its
    channels with the weights = (w0,w1,w2):
        grayscale[i,j] = w0*image[i,j,0] + w1*image[i,j,1] + w2*image[i,j,2]

    Parameters
    ----------
    image : array_like
        An array of shape (m,n,3) storing the RGB values for the color image.
    weights : tuple, optional
        A triple of weights (adding up to 1) to weight the image channels.

    Returns
    -------
    grayscale_image : ndarray
        The grayscale image where each pixel has a value in range [0.0,1.0].

    Raises
    ------
    ValueError
        If the image array is not a 3d array of size (m,n,3).

    """

    if (image.ndim != 2) and ((image.ndim != 3) or (image.shape[2] != 3)):
        raise ValueError("The image should be a 3d array of shape (m,n,3)!")

    grayscale_image = image @ weights

    if image.dtype == numpy.uint8:
        grayscale_image /= 2**8 - 1.0
    elif image.dtype == numpy.uint16:
        grayscale_image /= 2**16 - 1.0

    return grayscale_image
